/****************************************************************************
 *
 * (C) Copyright Gerd-Juergen Giefing 2010
 *     All Rights Reserved
 *
 ****************************************************************************
 *
 * @file      ar_start.c
 * @package   ARStarterKit
 * @author    Gerd-Juergen Giefing
 * @brief     Command line interface and reading, simple processing 
 *            and displaying of an image.
 *
 ****************************************************************************
 *
 * $Source:$
 * $Revision:$
 * $Date:$
 *
 * $Author:$
 * $Locker:$
 * $State:$
 *
 * $Log:$
 *
 ****************************************************************************/



/****************************************************************************
 * Declarations and variable definitions
 ****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>

#include <getopt.h>             /* getopt_long() */

#include <GL/gl.h>              /* for GLUT */
#include <GL/glu.h>
#include <GL/glut.h>

#include <rgb.h>
#include <ar.h>


static char inputname[256]       = "image.rgb";
static FILE *fp                  = NULL;

static int   img_width, img_height, img_frames;

static unsigned char *acq_image  = NULL;  /* block of image(s) */
static unsigned char *disp_image = NULL;  /* block of image(s) */

static int result = 1;

static const char short_options [] = "0:hu";
static const struct option
long_options [] = {
        { "input0",     required_argument,      NULL,           '0' },
        { "help",       no_argument,            NULL,           'h' },
        { 0, 0, 0, 0 }
};



/****************************************************************************
 * Functions
 ****************************************************************************/

void RGBKeyboardCheck( unsigned char key, int x , int y )
{

   switch (key) {
     case 'i':
       result = 0;
       break;
     case 'o':
       result = 1;
       break;
     case 'q':
     case 27:
       exit(0);
       break;
   }
   glutPostRedisplay();
}


void RGBReshapeImage(int width, int height)
{
 glViewport( 0, 0, width, height );
 glMatrixMode( GL_PROJECTION );
 glLoadIdentity();
 glOrtho( 0.0, width, 0.0, height, -1.0, 1.0 );
 glMatrixMode( GL_MODELVIEW );
 glLoadIdentity();
}


void RGBDisplayImage()
{
 char title[256] = "";

 sprintf(title, "AR_START   file: %s            (c)GJG 2010", inputname);
 glutSetWindowTitle(title);

 glDrawBuffer(GL_BACK);
 glReadBuffer(GL_BACK);

 /* draw image */
 glRasterPos2i(0, 0);
 glDisable(GL_DITHER);
 glPixelStorei(GL_UNPACK_ALIGNMENT, 1);

 if(result)
   glDrawPixels( img_width, img_height, GL_RGB, GL_UNSIGNED_BYTE, disp_image);
 else
   glDrawPixels( img_width, img_height, GL_RGB, GL_UNSIGNED_BYTE, acq_image);

 glFlush();
 glutSwapBuffers();

 return;
}


static void
usage                           (FILE *                 fp,
                                 int                    argc,
                                 char **                argv)
{
        fprintf (fp,
                 "Usage: %s [options]\n\n"
                 "Options:\n"
                 "-0 | --input0 name    Name of image pipe\n"
                 "-h | --help          Print this message\n"
                 "",
                 argv[0]);
}


int main (int argc, char **argv) {

 for (;;) {
   int index;
   int c;

   c = getopt_long (argc, argv,
                    short_options, long_options,
                    &index);

   if (-1 == c)
     break;

   switch (c) {
     case 0: /* getopt_long() flag */
       break;

     case '0':
       strcpy(inputname, optarg);
       break;

     case 'h':
       usage (stdout, argc, argv);
       exit (EXIT_SUCCESS);

     default:
       usage (stderr, argc, argv);
      exit (EXIT_FAILURE);
   }
 }
 if(argc != 1 && argc != 3) {

   usage (stdout, argc, argv);
   exit (EXIT_FAILURE);
 }

 /*======= Initialize GLUT ====================================*/
 glutInit( &argc , argv );
 glutInitDisplayMode( GLUT_RGBA | GLUT_DOUBLE );

 /*======= Read image file ==================================*/
 img_frames = readRGBFile(inputname, &acq_image, &img_width, &img_height);
 if(img_frames == 0) exit(0);
 img_frames = 1;

 /*======= Allocate out image buffer ==================================*/
 disp_image = (unsigned char *) malloc(img_width * img_height * 3);
 if(disp_image == NULL) exit(0);

 /*======= Process image file ==================================*/
 int x, y, c;

 for(y=0; y<img_height; y++) {
   for(x=0; x<img_width; x++) {
     for(c=0; c<3; c++) {

       if(acq_image[3*(img_width*y+x)+0] > 190 /* R value > 190                  */ &&
          acq_image[3*(img_width*y+x)+1] > 190 /* G value > 190                  */ &&
          acq_image[3*(img_width*y+x)+2] > 190 /* B value > 190                  */ &&
          y > 300 && y < 500                   /* y position between 300 and 500 */ &&
	  x > 200 && x < 400                   /* x position between 200 and 400 */   )

           /* mark pixel green */
           disp_image[3*(img_width*y+x)+1] = 255;
         else
           /* draw original pixel */
           disp_image[3*(img_width*y+x)+c] = acq_image[3*(img_width*y+x)+c];
     }
   }
 }
 x = 200;
 for(y=300; y<500; y++) {

   if(y>=0 && y<img_height)
     disp_image[3*(img_width*y+x)+2] = 255; /* draw vertical blue line*/
 }
 x = 400;
 for(y=300; y<500; y++) {

   if(y>=0 && y<img_height)
     disp_image[3*(img_width*y+x)+1] = 255; /* draw vertical green line*/
 }
 y = 300;
 for(x=200; x<400; x++) {

   if(x>=0 && x<img_width)
     disp_image[3*(img_width*y+x)+0] = 255; /* draw horizontal red line*/
 }
 y = 500;
 for(x=200; x<400; x++) {

   if(x>=0 && x<img_width)
     disp_image[3*(img_width*y+x)+0] = 255; /* draw horizontal yellow line*/
     disp_image[3*(img_width*y+x)+1] = 255;
 }

 /*======= Initialize window ==================================*/
 glutInitWindowPosition( 0, 0 );    /* initial window position */
 glutInitWindowSize( img_width, img_height );   /* set window size */
 glutCreateWindow( "AR START" );   /* create window with name */

 /*======= Initialize projection ==============================*/
 RGBReshapeImage(img_width, img_height);

 /*======= Initialize buffer ====================================*/
 glClearColor( 0.0 , 0.0 , 0.0 , 0.0 );
 glClear( GL_COLOR_BUFFER_BIT );
 glutSwapBuffers();
 glFlush();

 /*======= Register callback functions =========================*/
 glutDisplayFunc(  RGBDisplayImage );
 glutReshapeFunc(  RGBReshapeImage );
 glutKeyboardFunc( RGBKeyboardCheck );

 glutMainLoop();

 exit (EXIT_SUCCESS);
 return 0;
}

/****************************************************************************
 * End of file
 ****************************************************************************/
